 /*
  * util.c
  * Copyright (C) Aitor Cuadrado Zubizarreta <aitor_czr@gnuinos.org>
  * 
  * simple-netaid is free software: you can redistribute it and/or modify it
  * under the terms of the GNU General Public License as published by the
  * Free Software Foundation, either version 3 of the License, or
  * (at your option) any later version.
  * 
  * simple-netaid is distributed in the hope that it will be useful, but
  * WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.
  * 
  * You should have received a copy of the GNU General Public License along
  * with this program.  If not, see <http://www.gnu.org/licenses/>.
  * 
  * See the COPYING file.
  */
  
   
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h> 
#include <stdbool.h>     
#include <errno.h>
#include <sys/wait.h>  // waitpid
#include <spawn.h>

#include <simple-netaid/sbuf.h>
#include <simple-netaid/interfaces.h>
#include <simple-netaid/helpers.h>
#include <simple-netaid/wireless.h>
#include <simple-netaid/ipaddr.h>
#include <simple-netaid/ethlink.h>
#include <simple-netaid/iproute.h>

#include "util.h"
#include "config.h"
#include "def.h"
   
#define MAX_SIZE 256

int safe_execute_to_file(const char *pty_path, const char *out_path, bool append, char *const argv[])
{
    pid_t pid;
    int status;
    posix_spawn_file_actions_t actions;
    extern char **environ;

    posix_spawn_file_actions_init(&actions);
    posix_spawn_file_actions_addopen(&actions, STDERR_FILENO, "/dev/null", O_WRONLY, 0);

    if (pty_path) {
        posix_spawn_file_actions_addopen(&actions, STDIN_FILENO, pty_path, O_RDWR, 0);
    } else {
        posix_spawn_file_actions_addopen(&actions, STDIN_FILENO, "/dev/null", O_RDONLY, 0);
    }

    if (out_path) {
        int flags = O_CREAT | O_WRONLY | (append ? O_APPEND : O_TRUNC);
        posix_spawn_file_actions_addopen(&actions, STDOUT_FILENO, out_path, flags, 0644);
    } else if (pty_path) {
        posix_spawn_file_actions_adddup2(&actions, STDIN_FILENO, STDOUT_FILENO);
    } else {
        /* By default, output to limbo */
        posix_spawn_file_actions_addopen(&actions, STDOUT_FILENO, "/dev/null", O_WRONLY, 0);
    }

    if (posix_spawnp(&pid, argv[0], &actions, NULL, argv, environ) != 0) {
        posix_spawn_file_actions_destroy(&actions);
        return -1;
    }

    posix_spawn_file_actions_destroy(&actions);
    waitpid(pid, &status, 0);
    
    return WIFEXITED(status) ? WEXITSTATUS(status) : -1;
}

int safe_execute(const char *pty_path, char *const argv[])
{
    return safe_execute_to_file(pty_path, NULL, false, argv);
}
   
/**
 * \brief Create the directory to store the pidfile and the logfile.
 */
static void mkdir_piddir(const char *s)
{
    char *piddir, *tmp;

    if (!s || *s == '\0') return;

    piddir = strdup(s);
    if (!piddir) return;

    tmp = strrchr(piddir, '/');
    if (tmp) {
        *tmp = '\0';
        if (*piddir != '\0') {
            if (mkdir(piddir, 0755) == -1) {
                if (errno != EEXIST)
                    syslog(LOG_ERR, "Couldn't find PID's directory %s: %m", piddir);
            }
        }
    }
    free(piddir);
}
   
/**
 * \brief lock/unlock a file (option=1 locks the file; option=0 unlocks the file).
 * When trying to lock a file that is already locked, the function returns 1.
 */
int lock_file(const char *filename, int option)
{
    int fd;
    struct flock fl;

    /* Open with O_CLOEXEC for security reasons,
     * preventing children from inheriting the lock */
    fd = open(filename, O_RDWR | O_CREAT | O_CLOEXEC, 0640);
    if (fd < 0) {
        log_error("Unable to open lockfile %s: %m", filename);
        return -1;
    }

    memset(&fl, 0, sizeof(fl));
    /* option 1: Lock | option 0: Unlock */
    fl.l_type = (option == 1) ? F_WRLCK : F_UNLCK;
    fl.l_whence = SEEK_SET;
    fl.l_start = 0;
    fl.l_len = 0;  /* Block the entire file */

    /* Non-blocking blocking attempt */
    if (fcntl(fd, F_SETLK, &fl) == -1) {
        /* Is it already locked by other instance? */
        if (errno == EACCES || errno == EAGAIN) {
            close(fd);
            return 1;
        }
        close(fd);
        return -1;
    }

    if (option == 1) {
        /* We save it globally so that the lock persists */
        pid_fd = fd;
    } else {
        close(fd);
        if (option == 0) unlink(filename);
    }

    return 0;
}

/**
 * \brief Get the PID and check if the process is alive.
 */
pid_t get_pid_t(void)
{
    char txt[64];
    int fd;
    pid_t pid;
    ssize_t l;
    
    if (access(m_config->pidfile_path, F_OK) == -1)
        return (pid_t) -1;

    fd = open(m_config->pidfile_path, O_RDONLY);
    if (fd < 0) return (pid_t)-1;

    l = read(fd, txt, sizeof(txt) - 1);
    close(fd);

    if (l <= 0) return (pid_t)-1;
    
    txt[l] = '\0';
    pid = (pid_t)strtol(txt, NULL, 10);

    /* We verify whether the process actually exists by sending signal 0 */
    if (kill(pid, 0) != 0) {
        if (errno == ESRCH) {
            /* The process died, but the file remained there */
            syslog(LOG_WARNING, "Stuck PID file found. Cleaning up...");
            unlink(m_config->pidfile_path);
        }
        return (pid_t)-1;
    }

    return pid;
}

/**
 * \brief Kill the demon and wait for it to disappear.
 */
int wait_on_kill(int sig, int timeout_sec)
{
    pid_t pid = get_pid_t();
    if (pid < 0) return -1;

    if (kill(pid, sig) < 0)
        return -1;

    time_t end_time = time(NULL) + timeout_sec;

    while (time(NULL) < end_time) {
        /* If kill(pid, 0) fails with ESRCH, the process terminated */
        if (kill(pid, 0) < 0 && errno == ESRCH) {
            unlink(m_config->pidfile_path);
            if (m_config->lockfile_path) unlink(m_config->lockfile_path);
            return 0;
        }
        
        /* Short wait for 100ms using nanosleep */
        struct timespec ts = {0, 100000000L}; 
        nanosleep(&ts, NULL);
    }

    errno = ETIME;
    return -1;
}

void daemonize()
{
    pid_t pid;
    int fd;

    /* 1st fork */
    pid = fork();
    if (pid < 0) exit(EXIT_FAILURE);
    if (pid > 0) exit(EXIT_SUCCESS);

    /* Become a session leader */
    if (setsid() < 0)
        exit(EXIT_FAILURE);

    signal(SIGCHLD, SIG_IGN);

    /* 2nd fork */
    pid = fork();
    if (pid < 0)
        exit(EXIT_FAILURE);
    if (pid > 0)
        exit(EXIT_SUCCESS);

    umask(0);
    if (chdir("/tmp") != 0)
        exit(EXIT_FAILURE);

    /* Close standard descriptors safety */
    for (fd = sysconf(_SC_OPEN_MAX); fd >= 0; fd--)
        close(fd);

    fd = open("/dev/null", O_RDWR);
    if (fd != -1) {
        dup2(fd, STDIN_FILENO);
        dup2(fd, STDOUT_FILENO);
        dup2(fd, STDERR_FILENO);
        if (fd > 2)
            close(fd);
    }

    openlog(progname, LOG_PID, LOG_DAEMON);

    if (m_config->pidfile_path) {
        mkdir_piddir(m_config->pidfile_path);
      
        pid_fd = open(m_config->pidfile_path, O_RDWR | O_CREAT, 0640);
        if (pid_fd < 0) {
            syslog(LOG_ERR, "Cannot open pidfile %s: %m", m_config->pidfile_path);
            exit(EXIT_FAILURE);
        }
      
        // F_TLOCK: Fails if already locked, avoiding replicated instance */
        if (lockf(pid_fd, F_TLOCK, 0) < 0) {
            syslog(LOG_ERR, "Daemon already running (lock active)");
            exit(EXIT_FAILURE);
        }

        /* Delete previous content and write new PID */
        ftruncate(pid_fd, 0);
        dprintf(pid_fd, "%d\n", getpid());
        /* We do not close pid_fd; the lock remains while the process is active */
    }
}

void get_interfaces_simple(char *wired, size_t w_len, char *wireless, size_t wl_len) 
{
    DIR *dir;
    struct dirent *ent;
    char path[256];

    if (wired) wired[0] = '\0';
    if (wireless) wireless[0] = '\0';

    dir = opendir("/sys/class/net");
    if (!dir) return;

    while ((ent = readdir(dir)) != NULL) {
        // Ignore "." ".." and the loopback "lo"
        if (ent->d_name[0] == '.' || strcmp(ent->d_name, "lo") == 0 ||
                strncmp(ent->d_name, "bond", 4) == 0)
            continue;

        snprintf(path, sizeof(path), "/sys/class/net/%s/wireless", ent->d_name);
        
        if (access(path, F_OK) == 0) {
            if (wireless && wireless[0] == '\0')
                snprintf(wireless, wl_len, "%s", ent->d_name);
        } else {
            if (wired && wired[0] == '\0')
                snprintf(wired, w_len, "%s", ent->d_name);
        }
    }
    closedir(dir);
}

void ipaddr_flush_all_devices()
{
    char wired_if[IFNAMSIZ] = {0};
    char wireless_if[IFNAMSIZ] = {0};

    get_interfaces_simple(wired_if, sizeof(wired_if), wireless_if, sizeof(wireless_if));

    if (wired_if[0] != '\0')
        ipaddr_flush(wired_if);

    if (wireless_if[0] != '\0')
        ipaddr_flush(wireless_if);
}

int wired_connection(const char *ifname)
{ 
    if (!ifname)
        return -1;

    kill_all_processes();
    ipaddr_flush_all_devices();
    
    char *const ifdown_cmd[] = { "ifdown", "--force", (char *)ifname, NULL };
    safe_execute("/dev/null", ifdown_cmd);

    char *const check_cmd[] = { "ifquery", (char *)ifname, NULL };
    if (safe_execute("/dev/null", check_cmd) != 0) {
        log_error("Interface %s not defined in /etc/network/interfaces", ifname);
        return -1; 
    }
 
    char *const ifup_cmd[] = { "ifup", "-v", (char *)ifname, NULL };
    return safe_execute("/dev/null", ifup_cmd);
}

/**
 * \brief Wireless connection attempt
 */
int wireless_connection(const char *ifname)
{
    DIR *dir;
    struct dirent *ent;
    sbuf_t buffer;
    char path[PATH_MAX], best_path[PATH_MAX] = {0};
    char line[MAX_SIZE], best_ssid[128] = {0};
    int wstatus = -1, best_quality = -1;

    sbuf_init(&buffer);
    if (scan_active_wifis(ifname, &buffer) <= 0) {
        sbuf_free(&buffer);
        return -1;
    }

    dir = opendir(WIFI_DIR);
    if (!dir) {
        syslog(LOG_ERR, "Error: Unable to open %s: %m", WIFI_DIR);
        sbuf_free(&buffer);
        return -1;
    }

    while ((ent = readdir(dir))) {
        if (ent->d_name[0] == '.') continue;
        if (ent->d_type == DT_DIR) continue;

        snprintf(path, sizeof(path), "%s/%s", WIFI_DIR, ent->d_name);
        FILE *fp = fopen(path, "r");
        if (!fp) continue;

        while (fgets(line, sizeof(line), fp)) {
            char *start = strchr(line, '"');
            char *end = start ? strrchr(start + 1, '"') : NULL;

            if (start && end) {
                *end = '\0';
                char *ssid_in_file = start + 1;
                
                char *match = strcasestr(buffer.buf, ssid_in_file);
                if (match) {
                    char *pipe = strchr(match, '|');
                    if (pipe) pipe = strchr(pipe + 1, '|');
                    
                    if (pipe) {
                        int quality = atoi(pipe + 1);
                        if (quality > best_quality) {
                            best_quality = quality;
                            strlcpy(best_path, path, sizeof(best_path));
                            strlcpy(best_ssid, ssid_in_file, sizeof(best_ssid));
                        }
                    }
                }
            }
        }
        fclose(fp);
    }
    closedir(dir);
    sbuf_free(&buffer);

    if (*best_path == '\0') {
        syslog(LOG_WARNING, "No known networks were found available.");
        return -1;
    }

    syslog(LOG_INFO, "Better network: %s (Quality: %d). Connecting...",
            best_ssid, best_quality);

    int retries = 3;
    while (retries-- > 0) {
        kill_all_processes();
        ipaddr_flush(ifname);
        
        char *const down_args[] = { "ifdown", "-v", (char *)ifname, NULL };
        safe_execute("/dev/null", down_args);

        char *const wpa_args[] = { "wpa_supplicant", "-B", "-i",
                                    (char *)ifname, "-c", best_path, NULL };
        if (safe_execute("/dev/null", wpa_args) == 0) {
            sleep(2);
            char *const up_args[] = { "ifup", "-v", (char *)ifname, NULL };
            if (safe_execute("/dev/null", up_args) == 0) {
                syslog(LOG_INFO, "Successfull connection to %s", best_ssid);
                return 0;
            }
        }
        syslog(LOG_WARNING, "Failed attempt in %s. Retries: %d", best_ssid, retries);
        sleep(1);
    }

    return -1;
}
