  /*
  * policy.c
  * Copyright (C) Aitor Cuadrado Zubizarreta <aitor_czr@gnuinos.org>
  * 
  * simple-netaid is free software: you can redistribute it and/or modify it
  * under the terms of the GNU General Public License as published by the
  * Free Software Foundation, either version 3 of the License, or
  * (at your option) any later version.
  * 
  * simple-netaid is distributed in the hope that it will be useful, but
  * WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.
  * 
  * You should have received a copy of the GNU General Public License along
  * with this program.  If not, see <http://www.gnu.org/licenses/>.
  * 
  * See the COPYING file.
  */

#include <stdint.h>
#include <syslog.h>
#include <unistd.h>
#include <signal.h>
#include <getopt.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/wait.h>  // waitpid
#include <libgen.h>     // dirname() and basename()
#include <ctype.h>      // isdigit()	  
#include <errno.h>
#include <spawn.h>

#include <simple-netaid/sbuf.h>
#include <simple-netaid/ipaddr.h>
#include <simple-netaid/iproute.h>
#include <simple-netaid/interfaces.h>
#include <simple-netaid/helpers.h>
#include <simple-netaid/wireless.h>

#include "def.h"
#include "config.h"
#include "policy.h"
#include "util.h"
#include "iwlist.h"

#include <json.h>
#include <libubox/blobmsg.h>
#include <libubox/blobmsg_json.h>
#include <libubox/json_script.h>

#define WIFI_DIR "/etc/network/wifi"
#define PREFIX "/tmp/simple-netaid_"

#define PTY_MAX_LEN 16

#define strdup_or_null( str )  (str) != NULL ? strdup(str) : NULL

static struct ubus_request_data req_data;
static struct blob_buf bb_t;

/* Enum for EGI policy order */
enum {
	INTERFACE_DOWN_ID,
	INTERFACE_DOWN_IFNAME,
	__INTERFACE_DOWN_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy interface_down_policy[] =
{
	[INTERFACE_DOWN_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[INTERFACE_DOWN_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
};

/* Enum for EGI policy order */
enum {
	INTERFACE_UP_ID,
	INTERFACE_UP_IFNAME,
	__INTERFACE_UP_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy interface_up_policy[] =
{
	[INTERFACE_UP_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[INTERFACE_UP_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
};

/* Enum for EGI policy order */
enum {
	IFDOWN_ID,
	IFDOWN_IFNAME,
	IFDOWN_TTY,
	__IFDOWN_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy ifdown_policy[] =
{
	[IFDOWN_ID] = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[IFDOWN_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
	[IFDOWN_TTY] = { .name="tty", .type=BLOBMSG_TYPE_INT32 },
};

/* Enum for EGI policy order */
enum {
	IFUP_ID,
	IFUP_IFNAME,
	IFUP_TTY,
	__IFUP_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy ifup_policy[] =
{
	[IFUP_ID] = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[IFUP_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
	[IFUP_TTY] = { .name="tty", .type=BLOBMSG_TYPE_INT32 },
};

/* Enum for EGI policy order */
enum {
	IPADDR_FLUSH_ID,
	IPADDR_FLUSH_IFNAME,
	__IPADDR_FLUSH_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy ipaddr_flush_policy[] =
{
	[IPADDR_FLUSH_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[IPADDR_FLUSH_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
};

/* Enum for EGI policy order */
enum {
	DISCONNECT_ID,
	DISCONNECT_TTY,
	__DISCONNECT_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy disconnect_policy[] =
{
	[DISCONNECT_ID] = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[DISCONNECT_TTY] = { .name="tty", .type=BLOBMSG_TYPE_INT32 },
};

/* Enum for EGI policy order */
enum {
	WIRED_CONNECTION_ID,
	WIRED_CONNECTION_IFNAME,
	WIRED_CONNECTION_TTY,
	__WIRED_CONNECTION_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy wired_connection_policy[] =
{
	[WIRED_CONNECTION_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[WIRED_CONNECTION_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
	[WIRED_CONNECTION_TTY] = { .name="tty", .type=BLOBMSG_TYPE_INT32 },
};
 
/* Enum for EGI policy order */
enum {
	WPA_PASSPHRASE_ID,
	WPA_PASSPHRASE_ESSID,
	WPA_PASSPHRASE_PASSWD,
	WPA_PASSPHRASE_FILENAME,
	__WPA_PASSPHRASE_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy wpa_passphrase_policy[] =
{
	[WPA_PASSPHRASE_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[WPA_PASSPHRASE_ESSID] = { .name="essid", .type=BLOBMSG_TYPE_STRING },
	[WPA_PASSPHRASE_PASSWD] = { .name="passwd", .type=BLOBMSG_TYPE_STRING },
	[WPA_PASSPHRASE_FILENAME] = { .name="filename", .type=BLOBMSG_TYPE_STRING },
};
 
/* Enum for EGI policy order */
enum {
	WPA_SUPPLICANT_ID,
	WPA_SUPPLICANT_IFNAME,
	WPA_SUPPLICANT_FILENAME,
	WPA_SUPPLICANT_TTY,
	__WPA_SUPPLICANT_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy wpa_supplicant_policy[] =
{
	[WPA_SUPPLICANT_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[WPA_SUPPLICANT_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
	[WPA_SUPPLICANT_FILENAME] = { .name="filename", .type=BLOBMSG_TYPE_STRING },
	[WPA_SUPPLICANT_TTY]  = { .name="tty", .type=BLOBMSG_TYPE_INT32 },
};

/* Enum for EGI policy order */
enum {
	WIRELESS_CONNECTION_ID,
	WIRELESS_CONNECTION_IFNAME,
	WIRELESS_CONNECTION_ESSID,
	WIRELESS_CONNECTION_PASSWD,
	WIRELESS_CONNECTION_FILENAME,
	WIRELESS_CONNECTION_TTY,
	__WIRELESS_CONNECTION_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy wireless_connection_policy[] =
{
	[WIRELESS_CONNECTION_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[WIRELESS_CONNECTION_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
	[WIRELESS_CONNECTION_ESSID] = { .name="essid", .type=BLOBMSG_TYPE_STRING },
	[WIRELESS_CONNECTION_PASSWD] = { .name="passwd", .type=BLOBMSG_TYPE_STRING },
	[WIRELESS_CONNECTION_FILENAME] = { .name="filename", .type=BLOBMSG_TYPE_STRING },
	[WIRELESS_CONNECTION_TTY]  = { .name="tty", .type=BLOBMSG_TYPE_INT32 },
};

/* Enum for EGI policy order */
enum {
	UNINSTALL_SAVED_WIFI_ID,
	UNINSTALL_SAVED_WIFI_FILENAME,
	__UNINSTALL_SAVED_WIFI_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy uninstall_saved_wifi_policy[] =
{
	[UNINSTALL_SAVED_WIFI_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[UNINSTALL_SAVED_WIFI_FILENAME] = { .name="filename", .type=BLOBMSG_TYPE_STRING },
};

/* Enum for EGI policy order */
enum {
	SCAN_ACTIVE_WIFIS_ID,
	SCAN_ACTIVE_WIFIS_IFNAME,
	SCAN_ACTIVE_WIFIS_FORMAT,
	__SCAN_ACTIVE_WIFIS_MAX,
};
 
/* Ubus Policy */
const struct blobmsg_policy scan_active_wifis_policy[] =
{
	[SCAN_ACTIVE_WIFIS_ID]  = { .name="id", .type=BLOBMSG_TYPE_INT32 },
	[SCAN_ACTIVE_WIFIS_IFNAME] = { .name="ifname", .type=BLOBMSG_TYPE_STRING },
	[SCAN_ACTIVE_WIFIS_FORMAT]  = { .name="format", .type=BLOBMSG_TYPE_INT32 },
};

 
/* Ubus Methods */
const struct ubus_method netaid_methods[] =
{
	UBUS_METHOD("interface_down", interface_down_handler, interface_down_policy),
	UBUS_METHOD("interface_up", interface_up_handler, interface_up_policy),
	UBUS_METHOD("ifdown", ifdown_handler, ifdown_policy),
	UBUS_METHOD("ifup", ifup_handler, ifup_policy),
	UBUS_METHOD("ipaddr_flush", ipaddr_flush_handler, ipaddr_flush_policy),
	UBUS_METHOD("disconnect", disconnect_handler, disconnect_policy),
	UBUS_METHOD("wired_connection", wired_connection_handler, wired_connection_policy),
	UBUS_METHOD("wpa_passphrase", wpa_passphrase_handler, wpa_passphrase_policy),
	UBUS_METHOD("wpa_supplicant", wpa_supplicant_handler, wpa_supplicant_policy),
	UBUS_METHOD("wireless_connection", wireless_connection_handler, wireless_connection_policy),
	UBUS_METHOD("uninstall_saved_wifi", uninstall_saved_wifi_handler, uninstall_saved_wifi_policy),
	UBUS_METHOD("scan_active_wifis", scan_active_wifis_handler, scan_active_wifis_policy),
};
 
/* Ubus object type */
struct ubus_object_type netaid_obj_type =
	UBUS_OBJECT_TYPE("ering.netaid", netaid_methods);
 
/* Ubus object */
struct ubus_object netaid_obj=
{
	.name = "ering.netaid", 	/* with APP name */
	.type = &netaid_obj_type,
	.methods = netaid_methods,
	.n_methods = ARRAY_SIZE(netaid_methods),
	//.path= /* useless */
};

/**
 * \brief Callback function for ubus interface_down method handling
 */
int interface_down_handler(struct ubus_context *ctx, struct ubus_object *obj,
                           struct ubus_request_data *req, const char *method,
                           struct blob_attr *msg_t)
{
    char reply_msg[128];
    struct blob_attr *tb[__INTERFACE_DOWN_MAX];
    
    /* 1. Parseo y validación de argumentos */
    blobmsg_parse(interface_down_policy, ARRAY_SIZE(interface_down_policy), tb, blob_data(msg_t), blob_len(msg_t));

    if (!tb[INTERFACE_DOWN_IFNAME])
        return UBUS_STATUS_INVALID_ARGUMENT;

    const char *ifname = blobmsg_get_string(tb[INTERFACE_DOWN_IFNAME]);
   
    /* 2. Ejecución de la lógica de red */
    // interface_down() debe realizar la llamada al kernel (ioctl/netlink)
    interface_down(ifname);

    /* 3. Respuesta a Ubus usando buffer estático */
    snprintf(reply_msg, sizeof(reply_msg), "ip link set %s down", ifname);
 
    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "Server reply - Request has been proceeded", reply_msg);
    ubus_send_reply(ctx, req, bb_t.head);
 
    /* 4. Finalización de la petición */
    // Eliminamos defer_request ya que la operación es síncrona y atómica
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);

    return 0;
}

/**
 * \brief Callback function for ubus interface_up method handling
 */
int interface_up_handler(struct ubus_context *ctx, struct ubus_object *obj,
                         struct ubus_request_data *req, const char *method,
                         struct blob_attr *msg_t)
{
   char reply_msg[128];
   struct blob_attr *tb[__INTERFACE_UP_MAX];
    
   /* 1. Parseo y validación de argumentos */
   blobmsg_parse(interface_up_policy, ARRAY_SIZE(interface_up_policy), tb, blob_data(msg_t), blob_len(msg_t));

   if (!tb[INTERFACE_UP_IFNAME])
      return UBUS_STATUS_INVALID_ARGUMENT;

   // CORRECCIÓN: Usar el enum correcto para INTERFACE_UP
   const char *ifname = blobmsg_get_string(tb[INTERFACE_UP_IFNAME]);
   
   /* 2. Ejecución de la lógica de red */
   // Se asume que interface_up() ya realiza el 'ip link set ... up' internamente
   interface_up(ifname);

   

/* 3. Respuesta a Ubus sin sbuf_t */
   snprintf(reply_msg, sizeof(reply_msg), "ip link set %s up", ifname);
 
   blob_buf_init(&bb_t, 0);
   blobmsg_add_string(&bb_t, "Server reply - Request has been proceeded", reply_msg);
   ubus_send_reply(ctx, req, bb_t.head);
 
   /* 4. Finalización de la petición */
   ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
   
   return 0;
}

/**
 * \brief Callback function for ubus ifdown method handling
 */
int ifdown_handler(struct ubus_context *ctx, struct ubus_object *obj,
                   struct ubus_request_data *req, const char *method,
                   struct blob_attr *msg_t)
{
    int rc = 0;
    char pty[16];
    char reply_msg[64];
    bool redirect_ok = false;
    struct blob_attr *tb[__IFDOWN_MAX];
 
    /* 1. Parseo y validación de argumentos */
    blobmsg_parse(ifdown_policy, ARRAY_SIZE(ifdown_policy), tb, blob_data(msg_t), blob_len(msg_t));

    if (!tb[IFDOWN_IFNAME])
       return UBUS_STATUS_INVALID_ARGUMENT;

    const char *ifname = blobmsg_get_string(tb[IFDOWN_IFNAME]);   

    /* 2. Lógica: Solo intentar bajar si la interfaz tiene configuración activa */
    if (!ifquery(ifname)) 
       return 0; 
   
    /* 3. Configuración de terminal (PTY) */
    if (tb[IFDOWN_TTY] != NULL) {
        redirect_ok = true;
        snprintf(pty, sizeof(pty), "/dev/pts/%u", blobmsg_get_u32(tb[IFDOWN_TTY]));
    } else {
        strlcpy(pty, "/dev/null", sizeof(pty));
    }
   
    /* 4. Ejecución segura de ifdown --force */
    char *const ifdown_args[] = { "ifdown", "--force", (char *)ifname, NULL };
    rc = safe_execute(pty, ifdown_args);
   
    /* 5. Separador visual en terminal */
    char *const echo_args[] = { "echo", "           ------------            ", NULL };
    safe_execute(pty, echo_args);
   
    /* 6. Respuesta a UBUS */
    if (!redirect_ok) {
        snprintf(reply_msg, sizeof(reply_msg), "ifdown %s", ifname);
 
        blob_buf_init(&bb_t, 0);
        blobmsg_add_string(&bb_t, "Server reply - Request has been proceeded", reply_msg);
        ubus_send_reply(ctx, req, bb_t.head);
    }
   
    /* Finalización del ciclo de vida de la petición */
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
   
    return rc;
}

/**
 * \brief Callback function for ubus ifup method handling (Optimizado para ifupdown2)
 */
int ifup_handler(struct ubus_context *ctx, struct ubus_object *obj,
                 struct ubus_request_data *req, const char *method,
                 struct blob_attr *msg_t)
{
    int rc = -1;
    char pty[16];
    struct blob_attr *tb[__IFUP_MAX];
 
    /* 1. Parseo y validación */
    blobmsg_parse(ifup_policy, ARRAY_SIZE(ifup_policy), tb, blob_data(msg_t), blob_len(msg_t));

    if (!tb[IFUP_IFNAME])
        return UBUS_STATUS_INVALID_ARGUMENT;

    const char *ifname = blobmsg_get_string(tb[IFUP_IFNAME]);    

    /* 2. Configuración de terminal (PTY) */
    if (tb[IFUP_TTY] != NULL)
        snprintf(pty, sizeof(pty), "/dev/pts/%u", blobmsg_get_u32(tb[IFUP_TTY]));
    else
        strlcpy(pty, "/dev/null", sizeof(pty));
   
    /* 3. Ejecución síncrona */
    // En ifupdown2, safe_execute no retornará hasta que la interfaz esté lista
    char *const ifup_args[] = { "ifup", (char *)ifname, NULL };
    int wstatus = safe_execute(pty, ifup_args); 

    /* 4. Respuesta inmediata */
    blob_buf_init(&bb_t, 0);
    if (wstatus == 0) {
        rc = 0;
        blobmsg_add_string(&bb_t, "result", "success");
        blobmsg_add_string(&bb_t, "message", "Interface is up");
    } else {
        blobmsg_add_string(&bb_t, "result", "error");
        blobmsg_add_u32(&bb_t, "exit_code", wstatus);
    }

    ubus_send_reply(ctx, req, bb_t.head);
   
    // Ya no usamos defer_request porque la respuesta se envía aquí mismo
    return rc;
}

/**
 * \brief Callback function for ubus ipaddr_flush method handling
 */
int ipaddr_flush_handler(struct ubus_context *ctx, struct ubus_object *obj,
                         struct ubus_request_data *req, const char *method,
                         struct blob_attr *msg_t)
{
    // Buffers estáticos para evitar sbuf_t y fugas de memoria
    char wired_if[IFNAMSIZ] = {0};
    char wireless_if[IFNAMSIZ] = {0};

    struct blob_attr *tb[__IPADDR_FLUSH_MAX];
 
    /* 1. Parseo y validación crítica */
    blobmsg_parse(ipaddr_flush_policy, ARRAY_SIZE(ipaddr_flush_policy), tb, blob_data(msg_t), blob_len(msg_t));
   
    if (!tb[IPADDR_FLUSH_IFNAME])
        return UBUS_STATUS_INVALID_ARGUMENT;

    const char *ifname = blobmsg_get_string(tb[IPADDR_FLUSH_IFNAME]);

    /* 2. Obtener interfaces (Asumiendo que get_interfaces_simple es una versión que usa buffers) 
       Si get_interfaces requiere sbuf_t, asegúrate de liberarlos inmediatamente. */
    get_interfaces_simple(wired_if, sizeof(wired_if), wireless_if, sizeof(wireless_if));
   
    /* 3. Lógica de limpieza cruzada (Cross-flush) */
    // Si flusheamos wifi, reiniciamos cableada si está activa y viceversa
    if (wireless_if[0] && strcmp(ifname, wireless_if) == 0) {
        if (wired_if[0] && get_interface_status(wired_if)) {
            interface_down(wired_if);
            usleep(2500);
            interface_up(wired_if);
        }
    } else if (wired_if[0] && strcmp(ifname, wired_if) == 0) {
        if (wireless_if[0] && get_interface_status(wireless_if)) {
            interface_down(wireless_if);
            usleep(2500);
            interface_up(wireless_if);
        }
    }
   
    /* 4. Ejecución de limpieza */
    sleep(1);
    kill_all_processes();
    ipaddr_flush(ifname);

    /* 5. Responder a Ubus (Importante para no dejar al cliente esperando) */
    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "status", "flushed");
    ubus_send_reply(ctx, req, bb_t.head);

    return 0;
}

/**
 * \brief Callback function for ubus disconnect method handling
 */
int disconnect_handler(struct ubus_context *ctx, struct ubus_object *obj,
                       struct ubus_request_data *req, const char *method,
                       struct blob_attr *msg_t)
{
   int rc = 0;
   char pty[16];
   char reply_msg[128] = "Disconnected";
   int wstatus;
   bool redirect_ok = false;
   
   /* 1. Parseo de argumentos */
   struct blob_attr *tb[__DISCONNECT_MAX];
   blobmsg_parse(disconnect_policy, ARRAY_SIZE(disconnect_policy), tb, blob_data(msg_t), blob_len(msg_t));
   
   if (tb[DISCONNECT_TTY] != NULL) {
       redirect_ok = true;
       snprintf(pty, sizeof(pty), "/dev/pts/%u", blobmsg_get_u32(tb[DISCONNECT_TTY]));
   } else {
       strlcpy(pty, "/dev/null", sizeof(pty));
   }

   /* 2. Detección de Sistema de Init (Runit) */
   /*
   char buf[32] = {0};
   ssize_t sz = readlink("/proc/1/exe", buf, sizeof(buf) - 1);
   if (sz != -1) {
       buf[sz] = '\0'; // Asegurar nulo terminal
       if (strstr(buf, "runit")) {
           char *const stop_dhcp[] = { "sv", "stop", "dhcpcd", NULL };
           safe_execute(pty, stop_dhcp);
       }
   }
   */
   
   /* 3. Lógica de Desconexión de Interfaz */
   const char *ifname = iproute();
   if (ifname && *ifname != '0' && *ifname != '\0') {
       // Guardar mensaje de respuesta
       snprintf(reply_msg, sizeof(reply_msg), "Disconnected from %s", ifname);
   
       // Ejecutar ifdown forzado
       char *const ifdown_cmd[] = { "ifdown", "--force", (char *)ifname, NULL };
       safe_execute(pty, ifdown_cmd);
  
       // Limpieza de hardware y procesos
       interface_down(ifname);       
       ipaddr_flush(ifname);
       kill_all_processes();
       interface_up(ifname);
   } else {
       strlcpy(reply_msg, "No active connection to disconnect", sizeof(reply_msg));
   }
   
   /* 4. Respuesta a Ubus */
   if (!redirect_ok) {
       blob_buf_init(&bb_t, 0);
       blobmsg_add_string(&bb_t, "reply", reply_msg);
       ubus_send_reply(ctx, req, bb_t.head);
   }
    
   ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
   
   return rc;
}

/**
 * \brief Callback function for ubus wired_connection method handling
 */
int wired_connection_handler(struct ubus_context *ctx, struct ubus_object *obj,
                             struct ubus_request_data *req, const char *method,
                             struct blob_attr *msg_t)
{ 
   int rc = -1;
   char pty[16];
   char reply_msg[128] = {0};
   int wstatus;
   bool redirect_ok = false;
   
   // 1. Verificación de conexión activa
   const char *current_ip = iproute();
   if (current_ip && *current_ip != '0' && *current_ip != '\0') {
      snprintf(reply_msg, sizeof(reply_msg), "Already connected to %s", current_ip);
      goto Finish_wired_connection;
   }
   
   struct blob_attr *tb[__WIRED_CONNECTION_MAX];
   blobmsg_parse(wired_connection_policy, ARRAY_SIZE(wired_connection_policy), tb, blob_data(msg_t), blob_len(msg_t)); 
  
   if (!tb[WIRED_CONNECTION_IFNAME])
      return UBUS_STATUS_INVALID_ARGUMENT;

   const char *ifname = blobmsg_get_string(tb[WIRED_CONNECTION_IFNAME]);
   
   // 2. Configuración de terminal (PTY)
   if (tb[WIRED_CONNECTION_TTY] != NULL) {
       redirect_ok = true;
       snprintf(pty, sizeof(pty), "/dev/pts/%u", blobmsg_get_u32(tb[WIRED_CONNECTION_TTY]));
   } else {
       strlcpy(pty, "/dev/null", sizeof(pty));
   }
   
   // 3. Limpieza de interfaz
   if(get_interface_status(ifname)) {
      interface_down(ifname);
      usleep(2500);
      interface_up(ifname);
      sleep(1);
   }

   kill_all_processes();
   ipaddr_flush_all_devices();
   interface_down(ifname);
   
   // 4. Ciclo ifdown
   if (ifquery(ifname)) {
      char *const ifdown_cmd[] = { "ifdown", "-v", (char *)ifname, NULL };
      if (safe_execute(pty, ifdown_cmd) != 0)
         goto Finish_wired_connection;
   
      char *const echo_sep[] = { "echo", "           ------------            ", NULL };
      safe_execute(pty, echo_sep);
   }
   
    interface_up(ifname);
    sleep(1);

    // 5. Ciclo ifup
    if (!ifquery(ifname)) {
        char *const ifup_cmd[] = { "ifup", "-v", (char *)ifname, NULL };
        wstatus = safe_execute(pty, ifup_cmd);

        if (wstatus == 0) {
            rc = 0;
            snprintf(reply_msg, sizeof(reply_msg), "Interface %s is UP and configured", ifname);
        } else {
            snprintf(reply_msg, sizeof(reply_msg), "ifupdown2 failed (exit code %d)", wstatus);
        }
    }
    rc = 0;   
    char *const echo_sep[] = { "echo", "           ------------            ", NULL };
    safe_execute(pty, echo_sep);

Finish_wired_connection:
   // 6. Respuesta a UBUS
   blob_buf_init(&bb_t, 0);
   blobmsg_add_string(&bb_t, "reply", *reply_msg ? reply_msg : "Connection failed");
   ubus_send_reply(ctx, req, bb_t.head);
   
   ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
   
   return rc;
}

/**
 * \brief Callback function for ubus wpa_passphrase method handling
 */
int wpa_passphrase_handler(struct ubus_context *ctx, struct ubus_object *obj,
                           struct ubus_request_data *req, const char *method,
                           struct blob_attr *msg)
{
    struct blob_attr *tb[__WPA_PASSPHRASE_MAX];
    blobmsg_parse(wpa_passphrase_policy, __WPA_PASSPHRASE_MAX, tb, blob_data(msg), blob_len(msg));

    if (!tb[WPA_PASSPHRASE_ESSID] || !tb[WPA_PASSPHRASE_PASSWD] || !tb[WPA_PASSPHRASE_FILENAME])
        return UBUS_STATUS_INVALID_ARGUMENT;

    const char *essid = blobmsg_get_string(tb[WPA_PASSPHRASE_ESSID]);
    const char *passwd = blobmsg_get_string(tb[WPA_PASSPHRASE_PASSWD]);
    const char *path = blobmsg_get_string(tb[WPA_PASSPHRASE_FILENAME]);

    // Ejecución segura: genera el archivo con permisos 0600 directamente
    char *const args[] = { "wpa_passphrase", (char *)essid, (char *)passwd, NULL };
    int res = safe_execute_to_file("/dev/null", path, false, args);

    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "status", (res == 0) ? "created" : "failed");
    ubus_send_reply(ctx, req, bb_t.head);

    return 0;
}

/**
 * \brief Callback function for ubus wpa_supplicant method handling
 */
int wpa_supplicant_handler(struct ubus_context *ctx, struct ubus_object *obj,
                           struct ubus_request_data *req, const char *method,
                           struct blob_attr *msg_t)
{
    int rc = -1;
    char pty[PTY_MAX_LEN];      
    int wstatus;
    struct blob_attr *tb[__WPA_SUPPLICANT_MAX];
 
    blobmsg_parse(wpa_supplicant_policy, ARRAY_SIZE(wpa_supplicant_policy), tb, blob_data(msg_t), blob_len(msg_t));   
   
    // 1. Validación crítica de argumentos
    if (!tb[WPA_SUPPLICANT_IFNAME] || !tb[WPA_SUPPLICANT_FILENAME])
        return UBUS_STATUS_INVALID_ARGUMENT;

    const char *ifname = blobmsg_get_string(tb[WPA_SUPPLICANT_IFNAME]);
    const char *filename = blobmsg_get_string(tb[WPA_SUPPLICANT_FILENAME]);
   
    // 2. Configuración de terminal de salida (PTY)
    if (tb[WPA_SUPPLICANT_TTY]) {
        unsigned int tty_id = blobmsg_get_u32(tb[WPA_SUPPLICANT_TTY]);
        snprintf(pty, sizeof(pty), "/dev/pts/%u", tty_id);
    } else {
        strlcpy(pty, "/dev/null", sizeof(pty));
    }

    // 3. Preparar argumentos para ejecución directa
    char *const wpa_args[] = {
        "wpa_supplicant",
        "-B",               // Background
        "-i", (char *)ifname,
        "-c", (char *)filename,
        NULL
    };

    // 4. Ejecutar usando la función segura posix_spawnp
    wstatus = safe_execute(pty, wpa_args);
    
    if (wstatus == 0) {
        rc = 0;
    } else {
        syslog(LOG_ERR, "Fallo al iniciar wpa_supplicant en %s con config %s", ifname, filename);
    }

    // 5. Respuesta al cliente ubus
    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "result", (rc == 0) ? "success" : "failed");
    blobmsg_add_string(&bb_t, "command", "wpa_supplicant");
    ubus_send_reply(ctx, req, bb_t.head);

    // No es necesario defer si completamos aquí mismo
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);
   
    return rc;
}

/**
 * \brief Callback function for ubus wireless_connection method handling
 */
int wireless_connection_handler(struct ubus_context *ctx, struct ubus_object *obj,
                               struct ubus_request_data *req, const char *method,
                               struct blob_attr *msg_t)
{ 
    int rc = -1;
    char pty[16];      
    char path[1024];
    char reply_msg[256] = {0}; // Buffer para la respuesta final
    int wstatus = 0;
    bool pass_filter_ok = false;
   
    /* 1. Verificación de estado previo usando iproute() directo */
    const char *current_ip = iproute();
    if (current_ip && *current_ip != '0' && *current_ip != '\0') {
        snprintf(reply_msg, sizeof(reply_msg), "Already connected to %s", current_ip);
        goto Finish_wireless_connection;
    }
   
    struct blob_attr *tb[__WIRELESS_CONNECTION_MAX];
    blobmsg_parse(wireless_connection_policy, ARRAY_SIZE(wireless_connection_policy), tb, blob_data(msg_t), blob_len(msg_t));
    
    if (!tb[INTERFACE_UP_IFNAME] || !tb[WIRELESS_CONNECTION_ESSID])
        return UBUS_STATUS_INVALID_ARGUMENT;
  
    const char *ifname = blobmsg_get_string(tb[INTERFACE_UP_IFNAME]);
    const char *essid = blobmsg_get_string(tb[WIRELESS_CONNECTION_ESSID]);
    const char *passwd = tb[WIRELESS_CONNECTION_PASSWD] ? blobmsg_get_string(tb[WIRELESS_CONNECTION_PASSWD]) : "";
    const char *filename = tb[WIRELESS_CONNECTION_FILENAME] ? blobmsg_get_string(tb[WIRELESS_CONNECTION_FILENAME]) : "";
   
    /* 2. Configuración de terminal y rutas con snprintf */
    if (tb[WIRELESS_CONNECTION_TTY]) {
        snprintf(pty, sizeof(pty), "/dev/pts/%u", blobmsg_get_u32(tb[WIRELESS_CONNECTION_TTY]));
    } else {
        snprintf(pty, sizeof(pty), "/dev/null");
    }
	  
    if (filename && *filename != '\0') {
        snprintf(path, sizeof(path), "%s", filename);
    } else {
        // Generación de sufijo aleatorio simple sin sbuf
        unsigned int r = (unsigned int)time(NULL) % 10000;
        snprintf(path, sizeof(path), "/tmp/simple-netaid_%u", r);
    }
    
    /* 3. Validación de seguridad del path */
    char *tmp_path_ptr = strdup_or_null(path);
    if (tmp_path_ptr) {
        char *dname = dirname(tmp_path_ptr);
        if (dname && (strcmp(dname, "/etc/network/wifi") == 0 || strncmp(path, "/tmp/simple-netaid_", 19) == 0)) {
            pass_filter_ok = true;
        }
        free(tmp_path_ptr);
    }
   
    /* 4. Preparación de hardware */
    kill_all_processes();
    ipaddr_flush_all_devices();   
    interface_down(ifname);
    
    if (ifquery(ifname)) {
        char *const ifdown_cmd[] = { "ifdown", "-v", (char *)ifname, NULL };
        safe_execute(pty, ifdown_cmd);    
    }
   
    interface_up(ifname);
   
    /* 5. Generación de credenciales (Refactorizado) */
    if (pass_filter_ok && *passwd != '\0' && access(path, F_OK) != 0) {
        char *const pass_args[] = { "wpa_passphrase", (char *)essid, (char *)passwd, NULL };
        wstatus = safe_execute_to_file(pty, path, false, pass_args);
        if (wstatus != 0) goto Finish_wireless_connection;
    } 
    else if (*passwd == '\0' && (!filename || *filename == '\0')) {
        int fd = open(path, O_CREAT | O_WRONLY | O_TRUNC, 0600);
        if (fd >= 0) {
            dprintf(fd, "network={\n\tssid=\"%s\"\n\tkey_mgmt=NONE\n}\n", essid);
            close(fd);
        } else {
            goto Finish_wireless_connection;
        }
    }

    /* 6. Conexión y Activación */
    char *const wpa_args[] = { "wpa_supplicant", "-B", "-i", (char *)ifname, "-c", path, NULL };
    if (safe_execute(pty, wpa_args) != 0) goto Finish_wireless_connection;      
   
    sleep(1);

    // Ciclo ifup
    char *const ifup_cmd[] = { "ifup", "-v", (char *)ifname, NULL };
    wstatus = safe_execute(pty, ifup_cmd);

    if (wstatus == 0) {
        rc = 0;
        snprintf(reply_msg, sizeof(reply_msg), "Interface %s is UP and configured", ifname);
    } else {
        snprintf(reply_msg, sizeof(reply_msg), "ifupdown2 failed (exit code %d)", wstatus);
    }
    
    rc = 0;   
    char *const echo_sep[] = { "echo", "           ------------            ", NULL };
    safe_execute(pty, echo_sep);

Finish_wireless_connection:
    /* 7. Limpieza final de archivos temporales */
    if (rc != 0 && (!filename || *filename == '\0')) {
        unlink(path);
    }

    /* 8. Respuesta UBUS */
    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "reply", *reply_msg ? reply_msg : "Operation failed");
    ubus_send_reply(ctx, req, bb_t.head);
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);

    return rc;
}

/**
 * \brief Callback function for ubus uninstall_saved_wifi method handling
 */
int uninstall_saved_wifi_handler(struct ubus_context *ctx, struct ubus_object *obj,
                      struct ubus_request_data *req, const char *method,
                      struct blob_attr *msg_t)
{
    struct blob_attr *tb[__UNINSTALL_SAVED_WIFI_MAX];
    blobmsg_parse(uninstall_saved_wifi_policy, __UNINSTALL_SAVED_WIFI_MAX, tb, blob_data(msg_t), blob_len(msg_t));

    if (!tb[UNINSTALL_SAVED_WIFI_FILENAME])
        return UBUS_STATUS_INVALID_ARGUMENT;

    const char *filename = blobmsg_get_string(tb[UNINSTALL_SAVED_WIFI_FILENAME]);

    // Validación de seguridad: solo permitir borrar en el directorio wifi o tmp
    if (strstr(filename, "..") || (strncmp(filename, WIFI_DIR, strlen(WIFI_DIR)) != 0 && 
                                   strncmp(filename, PREFIX, strlen(PREFIX)) != 0)) {
        return UBUS_STATUS_PERMISSION_DENIED;
    }

    int res = unlink(filename);

    blob_buf_init(&bb_t, 0);
    blobmsg_add_string(&bb_t, "result", (res == 0) ? "deleted" : "failed");
    ubus_send_reply(ctx, req, bb_t.head);

    return 0;
}

/**
 * \brief Callback function for ubus scan_active_wifis method handling
 */
int scan_active_wifis_handler( struct ubus_context *ctx, struct ubus_object *obj,
			  struct ubus_request_data *req, const char *method,
			  struct blob_attr *msg_t )
{
    int rc = 0, format = 0;

    /* for parsed attr */
    struct blob_attr *tb[__SCAN_ACTIVE_WIFIS_MAX]; 
 
    /* Parse blob_msg from the caller to request policy */
    blobmsg_parse(scan_active_wifis_policy, ARRAY_SIZE(scan_active_wifis_policy), tb, blob_data(msg_t), blob_len(msg_t));
   
    if (tb[SCAN_ACTIVE_WIFIS_FORMAT] != NULL)
        format = blobmsg_get_u32(tb[SCAN_ACTIVE_WIFIS_FORMAT]);
   
    blob_buf_init(&bb_t, 0);
    int count = iwlist (blobmsg_data(tb[SCAN_ACTIVE_WIFIS_IFNAME]), &bb_t);
    ubus_send_reply(ctx, req, bb_t.head);
   
    ubus_defer_request(ctx, req, &req_data);
    ubus_complete_deferred_request(ctx, req, UBUS_STATUS_OK);

    return rc;
}
