/*
 * wireless.c
 * Copyright (C) Aitor Cuadrado Zubizarreta <aitor_czr@gnuinos.org>
 * 
 * This file is part of the Wireless Tools project
 *
 *      Jean II - HPLB '99 - HPL 99->07
 *  
 * released under the GPL license.
 *     Copyright (c) 1997-2007 Jean Tourrilhes <jt@hpl.hp.com>
 * 
 * simple-netaid is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * simple-netaid is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * See the COPYING file. 
 */

/*
 * This tool can access various piece of information on the card
 * not part of iwconfig...
 * You need to link this code against "-liw". 
 */

#include <sys/time.h>
#include <iwlib.h>

#include <simple-netaid/sbuf.h>

#include "iwlist.h"

static void *tbl;

/****************************** TYPES ******************************/

/*
 * Scan state and meta-information, used to decode events...
 */
typedef struct iwscan_state
{
  /* State */
  int			ap_num;		/* Access Point number 1->N */
  int			val_index;	/* Value in table 0->(N-1) */
} iwscan_state;

/**************************** CONSTANTS ****************************/

#define IW_SCAN_HACK		0x8000

// thread-safe and async-safe int to string for base 10
static
void itoa10_safe( int val, char* str ) {
   
   int rc = 0;
   int i = 0;
   int len = 0;
   int j = 0;
   bool neg = false;
   
   // sign check
   if( val < 0 ) {
      val = -val;
      neg = true;
   }
   
   // consume, lowest-order to highest-order
   if( val == 0 ) {
      str[i] = '0';
      i++;
   }
   else {
      while( val > 0 ) {
      
         int r = val % 10;
      
         str[i] = '0' + r;
         i++;
      
         val /= 10;
      }
   }
   
   if( neg ) {
      
      str[i] = '-';
      i++;
   }
   
   len = i;
   i--;
   
   // reverse order to get the number 
   while( j < i ) {
      
      char tmp = *(str + i);
      *(str + i) = *(str + j);
      *(str + j) = tmp;
      
      j++;
      i--;
   }
   
   str[len] = '\0';
}

/*------------------------------------------------------------------*/
/*
 * Display an Ethernet Socket Address in readable format.
 */
static inline char *
iw_saether_ntop(const struct sockaddr *sap, char* bufp)
{
  iw_ether_ntop((const struct ether_addr *) sap->sa_data, bufp);
  return bufp;
}

/***************************** SCANNING *****************************/
/*
 * This one behave quite differently from the others
 *
 * Note that we don't use the scanning capability of iwlib (functions
 * iw_process_scan() and iw_scan()). The main reason is that
 * iw_process_scan() return only a subset of the scan data to the caller,
 * for example custom elements and bitrates are ommited. Here, we
 * do the complete job...
 */

/*------------------------------------------------------------------*/
/*
 * Print one element from the scanning results
 */
static inline int
print_scanning_token(struct stream_descr *	stream,	/* Stream of events */
		     struct iw_event *		event,	/* Extracted token */
		     struct iwscan_state *	state,
		     struct iw_range *	iw_range,	/* Range info */
		     int		has_range,
		     struct blob_buf *buf)
{
  char		buffer[128];	/* Temporary buffer */

  /* Now, let's decode the event */
  switch(event->cmd)
    {
    case SIOCGIWAP:
      snprintf(buffer, sizeof(buffer), "Cell %02d", state->ap_num);
      tbl = blobmsg_open_table(buf, buffer);
      blobmsg_add_string(buf, "Address", iw_saether_ntop(&event->u.ap_addr, buffer));
      state->ap_num++;
      break;
    case SIOCGIWESSID:
      {
	char essid[4*IW_ESSID_MAX_SIZE+1];
	memset(essid, '\0', sizeof(essid));
	if((event->u.essid.pointer) && (event->u.essid.length))
	  iw_essid_escape(essid, event->u.essid.pointer, event->u.essid.length);
	if(event->u.essid.flags)
	  {
	    // Does it have an ESSID index ?
	    if((event->u.essid.flags & IW_ENCODE_INDEX) > 1)
	      {
	        sbuf_t s;
	        char seqbuf[10];
            seqbuf[9] = 0;
	        sbuf_init(&s);
            itoa10_safe (event->u.essid.flags & IW_ENCODE_INDEX, seqbuf);
	        sbuf_concat(&s, 4, essid, " [", seqbuf, "]");
	        blobmsg_add_string(buf, "Essid", s.buf);
	        sbuf_free(&s);
	      }
	    else
	      {
	        blobmsg_add_string(buf, "Essid", essid);
	      }
	  }
	else
	  {
	    blobmsg_add_string(buf, "Essid", "off/any/hidden");
      }
    blobmsg_close_table(buf, tbl);
      } 
      break;
      
    case SIOCGIWENCODE:
      {  
	unsigned char	key[IW_ENCODING_TOKEN_MAX];
	if(event->u.data.pointer)
	  memcpy(key, event->u.data.pointer, event->u.data.length);
	else
	  event->u.data.flags |= IW_ENCODE_NOKEY;
	if(event->u.data.flags & IW_ENCODE_DISABLED)
	  {
	    blobmsg_add_string(buf, "Encryption key", "off");
      }
	else
	  {
	    sbuf_t s;
	    sbuf_init(&s);
	    // Display the key
	    iw_print_key(buffer, sizeof(buffer), key, event->u.data.length, event->u.data.flags);
	    sbuf_addstr(&s, buffer); 

	    // Other info...
	    if((event->u.data.flags & IW_ENCODE_INDEX) > 1)
	      {
	        char seqbuf[10];
            seqbuf[9] = 0;
	        itoa10_safe (event->u.data.flags & IW_ENCODE_INDEX, seqbuf);
	        sbuf_concat(&s, 3, " [", seqbuf, "]");
	      }
	    if(event->u.data.flags & IW_ENCODE_RESTRICTED)
	      {
	        sbuf_addstr(&s, "   Security mode:restricted");
	      }  
	    if(event->u.data.flags & IW_ENCODE_OPEN)
	      {
	        sbuf_addstr(&s, "   Security mode:open");
	      } 
	    blobmsg_add_string(buf, "Encryption key", s.buf);
	    sbuf_free(&s);
	  }
      }
      break;
    case IWEVQUAL:
      if(has_range && (((&event->u.qual)->level != 0) || ((&event->u.qual)->updated & (IW_QUAL_DBM | IW_QUAL_RCPI))))
      {
        /* Deal with quality : always a relative value */
        if(!((&event->u.qual)->updated & IW_QUAL_QUAL_INVALID))
            snprintf(buffer, sizeof(buffer), "%d/%d", (&event->u.qual)->qual, iw_range->max_qual.qual);
      
	    blobmsg_add_string(buf, "Quality", buffer);
      }
      break;
    default:
   }	/* switch(event->cmd) */
   
   return(state->ap_num - 1);
}

/*------------------------------------------------------------------*/
/*
 * Perform a scanning on one device
 */
static int
print_scanning_info(int		skfd,
		    const char *	ifname,
		    struct blob_buf *buf)
{
  int count = 0;
  struct iwreq		wrq;
  struct iw_scan_req    scanopt;		/* Options for 'set' */
  int			scanflags = 0;		/* Flags for scan */
  unsigned char *	buffer = NULL;		/* Results */
  int			buflen = IW_SCAN_MAX_DATA; /* Min for compat WE<17 */
  struct iw_range	range;
  int			has_range;
  struct timeval	tv;				/* Select timeout */
  int			timeout = 15000000;		/* 15s */
   
   /* Get range stuff */
   has_range = (iw_get_range_info(skfd, ifname, &range) >= 0);

   /* Check if the interface could support scanning; otherwise, return doing nothing. */
   if((!has_range) || (range.we_version_compiled < 14)) return(-1);

   /* Init timeout value -> 250ms between set and first get */
   tv.tv_sec = 0;
   tv.tv_usec = 0; //250000;

   /* Clean up set args */
   memset(&scanopt, 0, sizeof(scanopt));
   wrq.u.data.pointer = (caddr_t) &scanopt;
   wrq.u.data.length = sizeof(scanopt);
   wrq.u.data.flags = 0;
    
    
   /* Initiate Scanning */
   if(iw_set_ext(skfd, ifname, SIOCSIWSCAN, &wrq) < 0)
   {
      if((errno != EPERM))
       {
         //fprintf(stderr, "%-8.16s  Interface doesn't support scanning : %s\n\n", ifname, strerror(errno));
         return(-1);
       }
       
      /* If we don't have the permission to initiate the scan, we may
       * still have permission to read left-over results.
       * But, don't wait !!! */
      tv.tv_usec = 0;
   }
 
   timeout -= tv.tv_usec;

  /* Forever */
  while(1)
    {
      fd_set		rfds;		/* File descriptors for select */
      int		last_fd;	/* Last fd */
      int		ret;

      /* Guess what ? We must re-generate rfds each time */
      FD_ZERO(&rfds);
      last_fd = -1;

      /* In here, add the rtnetlink fd in the list */

      /* Wait until something happens */
      ret = select(last_fd + 1, &rfds, NULL, NULL, &tv);

      /* Check if there was an error */
      if(ret < 0)
	{
	  if(errno == EAGAIN || errno == EINTR)
	    continue;
	  return(-1);
	}

      /* Check if there was a timeout */
      if(ret == 0)
	{
	  unsigned char *	newbuf;

	realloc:
	  /* (Re)allocate the buffer - realloc(NULL, len) == malloc(len) */
	  newbuf = realloc(buffer, buflen);
	  if(newbuf == NULL)
	    {
	      if(buffer)
		free(buffer);
	      return(-1);
	    }
	  buffer = newbuf;

	  /* Try to read the results */
	  wrq.u.data.pointer = buffer;
	  wrq.u.data.flags = 0;
	  wrq.u.data.length = buflen;
	  if(iw_get_ext(skfd, ifname, SIOCGIWSCAN, &wrq) < 0)
	    {
	      /* Check if buffer was too small (WE-17 only) */
	      if((errno == E2BIG) && (range.we_version_compiled > 16)
		 && (buflen < 0xFFFF))
		{
		  /* Some driver may return very large scan results, either
		   * because there are many cells, or because they have many
		   * large elements in cells (like IWEVCUSTOM). Most will
		   * only need the regular sized buffer. We now use a dynamic
		   * allocation of the buffer to satisfy everybody. Of course,
		   * as we don't know in advance the size of the array, we try
		   * various increasing sizes. Jean II */

		  /* Check if the driver gave us any hints. */
		  if(wrq.u.data.length > buflen)
		    buflen = wrq.u.data.length;
		  else
		    buflen *= 2;

		  /* wrq.u.data.length is 16 bits so max size is 65535 */
		  if(buflen > 0xFFFF)
		    buflen = 0xFFFF;

		  /* Try again */
		  goto realloc;
		}

	      /* Check if results not available yet */
	      if(errno == EAGAIN)
		{
		  /* Restart timer for only 100ms*/
		  tv.tv_sec = 0;
		  tv.tv_usec = 100000;
		  timeout -= tv.tv_usec;
		  if(timeout > 0)
		    continue;	/* Try again later */
		}

	      /* Bad error */
	      free(buffer);
	      fprintf(stderr, "%-8.16s  Failed to read scan data : %s\n\n",
		      ifname, strerror(errno));
	      return(-2);
	    }
	  else
	    /* We have the results, go to process them */
	    break;
	}

      /* In here, check if event and event type
       * if scan event, read results. All errors bad & no reset timeout */
    }

  if(wrq.u.data.length)
    {
      struct iw_event		iwe;
      struct stream_descr	stream;
      struct iwscan_state	state = { .ap_num = 1, .val_index = 0 };
      int			ret;
      iw_init_event_stream(&stream, (char *) buffer, wrq.u.data.length);
      
      int i=0;
      
      do
	{
	  /* Extract an event and print it */
	  ret = iw_extract_event_stream(&stream, &iwe, range.we_version_compiled);
	  if(ret > 0)
	    count = print_scanning_token(&stream, &iwe, &state, &range, has_range, buf);
	}
      while(ret > 0);
    }
  else
    printf("%-8.16s  No scan results\n\n", ifname);

  free(buffer);
  return(count);
}


/*------------------------------------------------------------------*/

int iwlist(const char *ifname, struct blob_buf *buf)
{
   int r = 0;
   int skfd;     /* generic raw socket desc. */

   /* Create a channel to the NET kernel. */
   if((skfd = iw_sockets_open()) < 0)
    {
      perror("Enable to create a channel to the NET kernel.");
      exit( EXIT_FAILURE );
    }
 
   r = print_scanning_info(skfd, ifname, buf);

   /* Close the socket. */
   iw_sockets_close(skfd);

  return r;
}
