 /*
  * iproute.c
  * Copyright (C) Aitor Cuadrado Zubizarreta <aitor@genuen.org>
  * 
  * Most of the code of this file is taken from the RTnetlink service routines
  * of the iproute2 project.
  * 
  * https://git.kernel.org/pub/scm/network/iproute2/iproute2.git/
  * 
  * simple-netaid is free software: you can redistribute it and/or modify it
  * under the terms of the GNU General Public License as published by the
  * Free Software Foundation, either version 3 of the License, or
  * (at your option) any later version.
  * 
  * simple-netaid is distributed in the hope that it will be useful, but
  * WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.
  * 
  * You should have received a copy of the GNU General Public License along
  * with this program.  If not, see <http://www.gnu.org/licenses/>.
  * 
  * See the COPYING file.
  */

#include <stdbool.h>  
#include "iproute.h"
#include "ll_map.h"
#include "def.h"

#include <sys/socket.h>
#include <arpa/inet.h>

#include <sys/ioctl.h>
#include <net/if.h>
#include <unistd.h>

#define IFNAME_SIZ 64


static struct {
	unsigned int tb;
	int cloned;
	char *flushb;
	int oif, oifmask;
	inet_prefix rvia;
	inet_prefix mdst;
	inet_prefix msrc;
} filter;

static char ifname[IFNAME_SIZ];	
static struct rtnl_handle rtnl_h = { .fd = -1 };
static int last_priority;  /* for persistence */

static int is_tunnel_device(const char *name)
{
    int sock = socket(AF_INET, SOCK_DGRAM, 0);
    if (sock < 0) return -1;

    struct ifreq ifr;
    memset(&ifr, 0, sizeof(ifr));
    strncpy(ifr.ifr_name, name, IFNAMSIZ - 1);

    if (ioctl(sock, SIOCGIFFLAGS, &ifr) < 0) {
        close(sock);
        return -1;
    }

    close(sock);

    // Key flags:
    // IFF_POINTOPOINT: Defines tunnel connections (Miredo, VPN)
    // IFF_NOARP: Common in tunnels since they do not require MAC resolution
    if (ifr.ifr_flags & IFF_POINTOPOINT)
        return 1;

    return 0;
}

char *iproute()
{	
    if (rtnl_open(&rtnl_h, 0) < 0)
        exit(1);
        
    int rcvbuf = 1024 * 1024; // 1MB
    if (setsockopt(rtnl_h.fd, SOL_SOCKET, SO_RCVBUF, &rcvbuf, sizeof(rcvbuf)) < 0)
        perror("setsockopt(SO_RCVBUF)");
    
    /* AF_UNSPEC (0) requests IPv4 and IPv6 simultaneously */
    int family = AF_UNSPEC;	
    
    memset(&ifname, 0, IFNAME_SIZ);
    
    // --- reset ---
    memset(ifname, 0, IFNAME_SIZ);
    memset(&filter, 0, sizeof(filter));

    last_priority = 0;
    
    filter.tb = RT_TABLE_MAIN;
	filter.mdst.bitlen = -1;
	filter.msrc.bitlen = -1;
	filter.oif = 0;
	if (filter.oif > 0)
		filter.oifmask = -1;

    /* We use wilddump to obtain the actual table (not the cache) */
    if (rtnl_wilddump_request(&rtnl_h, family, RTM_GETROUTE) < 0) {
        perror("Cannot send dump request");
        return NULL;
    }

    /* The third argument of rtnl_dump_filter_nc is the callback. 
     * Ensure that print_route returns 0 (i.e. continue) and not an error.
     * This function will execute print_route for each route received */
    struct rtnl_dump_filter_arg arg[] = {
        {
            .filter = (rtnl_filter_t)print_route,
            .arg1 = stdout, /* Our 'void *arg' in print_route */
        },
        { .filter = NULL, } /* Sentinel to indicate the end of the array */
    };

    if (rtnl_dump_filter_l(&rtnl_h, arg) < 0) {
        fprintf(stderr, "Dump terminated\n");
        rtnl_close(&rtnl_h);
        return NULL;
    }

    rtnl_close(&rtnl_h);

    return ifname;
}

unsigned int print_route(const struct sockaddr_nl *who, struct nlmsghdr *n)
{ 
    struct rtmsg *r = NLMSG_DATA(n);
    struct rtattr *tb[RTA_MAX + 1];
    
    memset(tb, 0, sizeof(tb));

    /* Calculate len using signed types to avoid circular overflows */
    ssize_t len = n->nlmsg_len - NLMSG_LENGTH(sizeof(*r));

    /* Security: if the message is short or not a route, abort silently */
    if (len < 0 || (n->nlmsg_type != RTM_NEWROUTE && n->nlmsg_type != RTM_DELROUTE)) {
        return 0;
    }

    /* Parse */
    parse_rtattr(tb, RTA_MAX, RTM_RTA(r), len);
	
	if (n->nlmsg_type != RTM_NEWROUTE && n->nlmsg_type != RTM_DELROUTE) {
		fprintf(stderr, _("Not a route: %08x %08x %08x\n"),
			n->nlmsg_len, n->nlmsg_type, n->nlmsg_flags);
		return -1;
	}

	if (filter.flushb && n->nlmsg_type != RTM_NEWROUTE)
		return 0;

    /* Omit routes that are not from the MAIN table (avoids local/loopback routes) */
    if (r->rtm_table != RT_TABLE_MAIN)
        return 0;
        
    /* Check kernel error flags */
    if (r->rtm_flags & (RTNH_F_DEAD | RTNH_F_LINKDOWN))
        return 0;

    /* Logic to detect "Internet" (Default via)
     * IPv4: rtm_dst_len == 0 and it does have Gateway
     * IPv6: rtm_dst_len == 0 and it does have Gateway or Interface */
    if (tb[RTA_OIF]) {
        int if_index = *(int *)RTA_DATA(tb[RTA_OIF]);
        const char *name = ll_index_to_name(if_index);

        if (name) {
            int tunnel_status = is_tunnel_device(name);
            
            /* The device is a tunnel (Point-to-Point) */
            if (tunnel_status > 0) {
                    last_priority = 2;
                    strlcpy(ifname, name, sizeof(ifname));
                    return 1;             
             /* We only accept the physical interface if we 
              * haven't found a tunnel yet (Ethernet/WiFi) */
            } else if (tunnel_status == 0 && last_priority < 2) {
                
                /* General IPv6 logic */
                if (r->rtm_family == AF_INET6) {
                    /* In IPv6, if it is the MAIN table and has a global destination (len 0)
                     * It is considered an exit route even if it does not have a gateway
                     * (e.g. fe80::...) */
                    if (r->rtm_dst_len == 0 && (tb[RTA_GATEWAY] || tb[RTA_OIF])) {
                        last_priority = 1;
                        strlcpy(ifname, name, sizeof(ifname));
                        return 1;
                    }
                /* General IPv4 logic */
                } else if (r->rtm_family == AF_INET) {
                    if (r->rtm_dst_len == 0 && tb[RTA_GATEWAY]) {
                        last_priority = 1;
                        strlcpy(ifname, name, sizeof(ifname));
                        return 1;
                    }
                }
                return 0;
            }
        }
    }
    
    return 0;
}

int rtnl_rtcache_request(struct rtnl_handle *rtnl_h, int family)
{
	struct {
        struct nlmsghdr nlh;
        struct rtmsg rtm;
    } req = {
        .nlh.nlmsg_len = sizeof(req),
        .nlh.nlmsg_type = RTM_GETROUTE,
        .nlh.nlmsg_flags = NLM_F_DUMP | NLM_F_REQUEST,
        .nlh.nlmsg_seq = ++rtnl_h->seq,
        .rtm.rtm_family = family,
        .rtm.rtm_flags = 0,   /* only real routes, not a cache */
    };

	struct sockaddr_nl nladdr = {
        .nl_family = AF_NETLINK,
        .nl_groups = RTMGRP_IPV4_ROUTE | RTMGRP_IPV6_ROUTE,
    };

	return sendto(rtnl_h->fd, (void *)&req, sizeof(req), 0,
                    (struct sockaddr *)&nladdr, sizeof(nladdr));
}
